/* 
 *         GA Net - a genetic algorithm for generating Network Intusion Rules
 *
 *       Copyright (C) 2010 Brian E. Lavender <brian@brie.com> 
 *
 *                     http://www.brie.com/brian/ganet/ 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <stdlib.h>
#include "types.h"
#include "compare.h"
#include "print.h"
#include "service_attacks.h"

#ifndef SWAP_4
#define SWAP_4(x) ( ((x) << 24) | \
         (((x) << 8) & 0x00ff0000) | \
         (((x) >> 8) & 0x0000ff00) | \
         ((x) >> 24) )
#endif 



// Idx Feature Name    Format Number of Genes
//              byte  0 1 2 3
//  0  Duration         h:m:s         3
//  1  Protocol         Int           1
//  2  Source_port      Int           1
//  3  Destination_port Int           1
//                byte  0 1 2 3
//  4  Source_IP        a.b.c.d       4
//                byte  0 1 2 3
//  5  Destination_IP   a.b.c.d       4
//  6  Attack_name      Int           1
//
// Chromosome length 7

// myEvolve - individual from evovlved data
// myAudit - individual from Audit data
// return match variable
// 0 - no match
// 1 - match
gboolean compare_a(individual *trainer, individual *myAudit) {
  // assume we have a match
  int match = TRUE;
  int i, j;
  
  time_stamp tmpTimeE, tmpTimeA;
  //  IPAddr tmpIPE, tmpIPA;


  //  g_printf("match is %d\n",match);
  
  for (i=0; i<6; i++) {
    switch (i) {
    case 0: // Duration
    case 4: // Source IP
    case 5: // Destination IP
      // PART 0 of chromosome - Duration
      //  g_printf("Chrome %d\n",i);
      tmpTimeE.tot = trainer->chrome[i];
      tmpTimeA.tot = myAudit->chrome[i];

      // g_printf("a tot %x\n", tmpTimeE.tot);
      // g_printf("b tot %x\n", tmpTimeA.tot);
      // Assumes that the first byte of duration is -1
      for (j = 0; j<4; j++) {
	// We want to see if it doesn't match.
	if ( ! 
	     ( tmpTimeE.byte[j] == -1 || tmpTimeE.byte[j] == tmpTimeA.byte[j] ) 
	     ) 
	  match = FALSE;

	/* 	g_printf("chrome %d match is %d %d %d\n",i,match, */
	/* 		 tmpTimeE.byte[j], */
	/* 		 tmpTimeA.byte[j]); */
      }
      break;

    case 1: // Protocol
    case 2: // Source Port
    case 3: // Dest Port
      
      if ( ! 
	   ( trainer->chrome[i] == -1 || trainer->chrome[i] == myAudit->chrome[i] ) 
	   ) 
	match = FALSE;
      // g_printf("chrome %d match is %d\n",i,match);
      break;
    default:
      ;
    }
  }
    
  return match;
}

gboolean compare_b(individual *trainer, individual *myAudit) {
  // assume we have a match
  gboolean match = TRUE;
  if ( ! 
       //       ( trainer->chrome[G_ATTACK] == -1 || trainer->chrome[G_ATTACK] == myAudit->chrome[G_ATTACK] ) 
       (  trainer->chrome[G_ATTACK] == myAudit->chrome[G_ATTACK] ) 
       ) 
    match = FALSE;
  return match;
      
}


gint get_mag(GSList *auditList, individual *trainer, guint *mag_AandB,
	     guint *magA ) {
  GSList *iterator = NULL;
  individual *auditItem;
  gint n = 0;

  for (iterator = auditList; iterator; iterator = iterator->next) {
    auditItem = (individual*)iterator->data;
    

    if ( compare_a(trainer,auditItem) && compare_b(trainer,auditItem) )
      (*mag_AandB)++;

    if ( compare_a(trainer,auditItem) ) 
      (*magA)++;
    n++;
  }

  return n;

}

void get_fitness(GSList *auditList, individual *trainer, gdouble w1,
		 gdouble w2, guint N) {
  guint countAandB=0,  countA=0;
  double fitness ; 
  double support;
  double confidence;
  // Check the training individual
  get_mag(auditList, trainer, &countAandB, &countA);

  // Must not divide by 0
  if ( countA > 0 && N > 0 ) {
    support = countAandB / (double)N; 
    confidence = countAandB / (double) countA;
    fitness = w1 * support + w2 * confidence;
  } else 
    fitness = 0.0; 
  // Assign the fitness
  trainer->fitness = fitness;
  set_string_individual(trainer);
}

gint sort_function(gconstpointer a, gconstpointer b) {
  individual **pia, **pib;

  gdouble fitness_a, fitness_b, delta;
  pia = (individual **) a;
  pib = (individual **) b;
  
  fitness_a = (*pia)->fitness;
  fitness_b = (*pib)->fitness;
  delta = fitness_a - fitness_b;

  //  g_print("fitness a: %.4f b: %.4f\n", fitness_a, fitness_b);

  if ( delta < 0.001 ) // they are equal
    return 0;

  if ( fitness_a < fitness_b )
    return -1;
  else
    return 1;
}    

gint sort_functionV2(gconstpointer a, gconstpointer b) {
  individual **pia, **pib;

  gdouble fitness_a, fitness_b, delta;
  pia = (individual **) a;
  pib = (individual **) b;
  
  fitness_a = (*pia)->fitness;
  fitness_b = (*pib)->fitness;
  delta = fitness_a - fitness_b;

  //  g_print("fitness a: %.4f b: %.4f\n", fitness_a, fitness_b);

  if ( (*pia)->chrome[G_ATTACK] < (*pib)->chrome[G_ATTACK] )
    return -1;
  else if ( (*pia)->chrome[G_ATTACK] > (*pib)->chrome[G_ATTACK] )
    return 1;
  else {

/*     if ( delta < 0.001 ) // they are equal */
/*       return 0; */

    if ( fitness_a < fitness_b ) {
      return 1;
    } else if ( fitness_a > fitness_b ) {
      return -1;
    } 

    return 0;

  }
  // Should not get here
}    

void destroyInd(gpointer myInd) {
  individual *pInd;
  gdouble fitnessInd;
  pInd = (individual *)myInd;
  fitnessInd = pInd->fitness;
  g_slice_free(individual, pInd );
  global_individual_count--;

}

void normalize(gint *a, gint*b) {
  gint tmp;

  if ( *a > *b ) {
    tmp = *a;
    *a = *b;
    *b = tmp;
  }
}


gint get_crossByte(guint randInt) {
  gint idx;
  gint offset;
  guint base;
  gint rValue = -1;

  switch(randInt) {

  case 0: // left edge of chromosome storage
  case 1: // left edge of chromosome
  case 2:
  case 3:
    base = 0;
    idx = 0;
    break;
  case 4:
    base = 4;
    idx = 1;
    break;
  case 5:
    base = 5;
    idx = 2;
    break;
  case 6:
    base = 6;
    idx = 3;
    break;
  case 7:
  case 8:
  case 9:
  case 10:
    base = 7;
    idx = 4;
    break;
  case 11:
  case 12:
  case 13:
  case 14:
    base = 11;
    idx = 5;
    break;
  case 15:
    base = 15;
    idx = 6;
    break;
  case 16: // right edge of chromosome
    base = 16;
    idx = 7; // 
    break;
    
  default:
    base = 0;
    idx = 0;
    ;
    //g_print("Default\n");
  }

  //g_print("randInt is %d\n",randInt);
  
  offset = randInt - base;
  rValue = idx * 4 + offset;


    return rValue;
}

void breed_v1(GRand *rnd, individual *parent1, individual *parent2, 
	   individual *child1, individual *child2 ) {

  gint randInt, whichbyte;

  // Pick a random integer between [1,17)
  randInt = g_rand_int_range(rnd,1,17);

  whichbyte = get_crossByte(randInt );

  
  if ( whichbyte == 1 || whichbyte == 28 ) {
    //g_print("No crossover\n");
    g_memmove(child1->chrome,parent1->chrome, NUM_GENE*4 );
    g_memmove(child2->chrome,parent2->chrome, NUM_GENE*4 );
  }
  else {

    g_memmove(child1->chrome,parent1->chrome, whichbyte );
    g_memmove((char *)(child1->chrome) + whichbyte ,
	      (char *)(parent2->chrome) + whichbyte ,
	      NUM_GENE*4 - whichbyte );

    g_memmove(child2->chrome,parent2->chrome, whichbyte );
    g_memmove((char *)(child2->chrome) + whichbyte ,
	      (char *)(parent1->chrome) + whichbyte ,
	      NUM_GENE*4 - whichbyte );


  }

}

void breed_v2(GRand *rnd, individual *parent1, individual *parent2, 
	   individual *child1, individual *child2 ) {

  gint randInt, whichbyte;
  individual t1, t2;
  gboolean putBack = FALSE;

  // If parents are different types, don't cross over certain data

  if (parent1->chrome[G_ATTACK] != parent2->chrome[G_ATTACK]) {
    // Stash away parent data
    g_memmove(&t1, parent1, sizeof(individual));
    g_memmove(&t2, parent2, sizeof(individual));
    putBack = TRUE;
  }

  // Pick a random integer between [1,17)
  randInt = g_rand_int_range(rnd,1,17);

  whichbyte = get_crossByte(randInt );
  
  if ( whichbyte == 1 || whichbyte == 28 ) {
    //g_print("No crossover\n");
    g_memmove(child1->chrome,parent1->chrome, NUM_GENE*4 );
    g_memmove(child2->chrome,parent2->chrome, NUM_GENE*4 );
  }
  else {
    g_memmove(child1->chrome,parent1->chrome, whichbyte );
    g_memmove((char *)(child1->chrome) + whichbyte ,
	      (char *)(parent2->chrome) + whichbyte ,
	      NUM_GENE*4 - whichbyte );

    g_memmove(child2->chrome,parent2->chrome, whichbyte );
    g_memmove((char *)(child2->chrome) + whichbyte ,
	      (char *)(parent1->chrome) + whichbyte ,
	      NUM_GENE*4 - whichbyte );

  }

  // putBack if true
  if (putBack) {
    // Fix back up child 1.
    child1->chrome[G_DEST_IP] = t1.chrome[G_DEST_IP];
    child1->chrome[G_SERVICE] = t1.chrome[G_SERVICE];
    child1->chrome[G_ATTACK] = t1.chrome[G_ATTACK];

    // Fix back up child 2.
    child2->chrome[G_DEST_IP] = t2.chrome[G_DEST_IP];
    child2->chrome[G_SERVICE] = t2.chrome[G_SERVICE];
    child2->chrome[G_ATTACK] = t2.chrome[G_ATTACK];
  }

}

void breed_midpoint(GRand *rnd, individual *parent1, individual *parent2, 
		    individual *child1, individual *child2 ) {

  gint randInt1, randInt2;
  gint whichbyte1, whichbyte2, delta1, delta2, delta3;
  randInt1 = g_rand_int_range(rnd, 1,17);
  randInt2 = g_rand_int_range(rnd, 1,17);
  
  whichbyte1 = get_crossByte(randInt1);
  whichbyte2 = get_crossByte(randInt2);

  // Make sure that whichbyte2 is greater than whichbyte1
  normalize(&whichbyte1, &whichbyte2);

  //  g_print("Bytes  %d %d\n", whichbyte1, whichbyte2);

  delta1 =  whichbyte1;
  delta2 = whichbyte2 - whichbyte1;
  delta3 = NUM_GENE*4 - whichbyte2;
  

  if (delta1 < 0 || delta2 < 0 || delta3 < 0) {
    g_print("Negative delta! %d %d %d\n",delta1, delta2, delta3);
    exit (-1);
  }

  // Child 1
  if (delta1 > 0)
    g_memmove((char *)(child1->chrome), (char *)(parent1->chrome), delta1 );
  
  if (delta2 >0)
    g_memmove((char *)(child1->chrome) + whichbyte1 ,
	      (char *)(parent2->chrome) + whichbyte1  ,
	      delta2 );

  if (delta3 > 0)
    g_memmove((char *)(child1->chrome) + whichbyte2, 
	      (char *)(parent1->chrome) + whichbyte2, 
	      delta3 );

  // Child 2
  if (delta1 > 0)
    g_memmove(child2->chrome, parent2->chrome, delta1 );

  if (delta2 > 0)
  g_memmove((char *)(child2->chrome) + whichbyte1 ,
	    (char *)(parent1->chrome) + whichbyte1  ,
	    delta2 );

  if (delta3 > 0)
    g_memmove((char *)child2->chrome + whichbyte2, 
	      (char *)parent2->chrome + whichbyte2, 
	      delta3 );

}
