/* 
 *         GA Net - a genetic algorithm for generating Network Intusion Rules
 *
 *       Copyright (C) 2010 Brian E. Lavender <brian@brie.com> 
 *
 *                     http://www.brie.com/brian/ganet/ 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "types.h"
#include "print.h"
#include "service_attacks.h"
#include "rand.h"

#define BUF_SZ 80


#ifndef SWAP_4
#define SWAP_4(x) ( ((x) << 24) |		\
		    (((x) << 8) & 0x00ff0000) | \
		    (((x) >> 8) & 0x0000ff00) | \
		    ((x) >> 24) )
#endif


// returns an array index 
// input 
// numUniqueP1 - number of elements plus the wildcard element. 
//               The wildcard element is assumed to be in index 0.
//               {-1,5,3,98,34,4}
// wcardProb   - Probability that you want the wildcard chosen.
guint randslot(GRand *rnd, guint numUniqueP1, double wcardProb ) {
  gdouble indProb;
  gdouble randVal;
  guint slot;
  
  indProb = (1.0 - wcardProb )/ ( numUniqueP1 - 1 );
  //  g_printf("Individual probability minus WCARD is %.6f\n",indProb);

  // Get a random number between [0,1)
  randVal = g_rand_double(rnd);

  if (randVal < wcardProb ) {
    //g_printf("Got a wildcard\n");
    slot = 0;
  } else {
    slot =  (randVal - wcardProb)/ ( 1 - wcardProb) * (numUniqueP1 - 1) + 1;

  }
  return slot;
}



void makeRandIndV1(GRand *rnd, individual **myInd)
{
  int tmp;
  time_stamp myDuration;
  guint tmpChrome[NUM_GENE];


  // Create a random chromosome
  
  // Time
  myDuration.tot = g_rand_int(rnd);
  myDuration.byte[0] = 0xff; // zero out first byte

  // copy myDuration to tmpChrome
  tmpChrome[0] = myDuration.tot; 

  // protocol Loock at header file for protocol definitions
  // TODO - allow for a wildcard
  tmpChrome[1] = g_rand_int_range(rnd, 0, ENDP); 

  // Src port. Max port number is range is [1, 2^16 -1 ]
  tmpChrome[2] = g_rand_int_range(rnd,1,0x10000 ); 

  // Dst port. Max port number is range [1 , 2^16 -1 ]
  tmpChrome[3] = g_rand_int_range(rnd,1,0x10000 ); 

  // full range on source IP
  tmpChrome[4] = g_rand_int(rnd); 

  // full range on dest IP address
  tmpChrome[5] = g_rand_int(rnd); 

  // random number [ 0 , END_A ]
  // which attack should this match?
  tmp = g_rand_int_range(rnd, 0, END_A ); 
  tmpChrome[6] = tmp;

  
  // Malloc an individual
  *myInd = g_slice_new0(individual);
  global_individual_count++;
  //*myInd = (individual *)g_malloc0(sizeof(individual));


  // Copy the temp chromosome into the individual
  g_memmove((*myInd)->chrome, tmpChrome, 4 * NUM_GENE);
  g_snprintf((*myInd)->desc, DESC_SZ,  "Training Chromosome" );


  
}


void makeRandIndV2(GRand *rnd, double wcardProb, individual **myInd,
		   GArray *garraysL[NUM_HTABLES], 
		   GArray *garraysC[NUM_HTABLES][SUBH] ) {
  // TODO: stub for code
  gint i; // loop counter
  guint tmpChrome[NUM_GENE]; // tempChrome created
  time_stamp myT; // temporary time stamp data
  IPAddr myIP; // temporary IP address
  guint mySlot; // random slot number picked
  
  //  time_stamp myT; // temporary time stamp data
  // IPAddr myIP; // temporary IP address  
  myT.byte[0] = 0xff; // not used, all should be -1

  // Hours, Minutes, Seconds
  for (i=1; i<4; i++) {
    mySlot = randslot(rnd, garraysC[G_DURATION][i]->len, wcardProb);
    myT.byte[i] = g_array_index (garraysC[G_DURATION][i], guchar, mySlot);
  }

  tmpChrome[G_DURATION] = myT.tot; // duration

  // Service
  mySlot = randslot(rnd, garraysL[G_SERVICE]->len, wcardProb);
  tmpChrome[G_SERVICE] = g_array_index (garraysL[G_SERVICE], guint, mySlot);

  // Source Port
  mySlot = randslot(rnd, garraysL[G_SOURCE_PORT]->len, wcardProb);
  tmpChrome[G_SOURCE_PORT] = g_array_index (garraysL[G_SOURCE_PORT], guint, mySlot);

  // Dest Port
  mySlot = randslot(rnd, garraysL[G_DEST_PORT]->len, wcardProb);
  tmpChrome[G_DEST_PORT] = g_array_index (garraysL[G_DEST_PORT], guint, mySlot);

  // Source IP xxx.xxx.xxx.xxx
  //            0   1   2   3
  for (i=0; i<4; i++) {
    mySlot = randslot(rnd, garraysC[G_SRC_IP][i]->len, wcardProb);
    myIP.octet[i] = g_array_index (garraysC[G_SRC_IP][i], guchar, mySlot);
  }
  tmpChrome[G_SRC_IP] = myIP.full; // source IP

  // Dest IP xxx.xxx.xxx.xxx
  //          0   1   2   3
  for (i=0; i<4; i++) {
    mySlot = randslot(rnd, garraysC[G_DEST_IP][i]->len, wcardProb);
    myIP.octet[i] = g_array_index (garraysC[G_DEST_IP][i], guchar, mySlot);
  }
  tmpChrome[G_DEST_IP] = myIP.full; // dest IP

  // Attack. Do not select wildcard, so put its probability at 0.
  mySlot = randslot(rnd, garraysL[G_ATTACK]->len, 0.0 );
  tmpChrome[G_ATTACK] = g_array_index (garraysL[G_ATTACK], guint, mySlot);

  *myInd = g_slice_new0(individual);
  global_individual_count++;

  // Copy the temp chromosome into the individual
  g_memmove( (*myInd)->chrome, tmpChrome, 4 * NUM_GENE);
  g_snprintf((*myInd)->desc, DESC_SZ,  "Training Chromosome" );

}


individual* makeEmptyInd(void) {
  individual *a;
  a = g_slice_new0(individual);
  global_individual_count++;
  return a;
}

void swapPop(GPtrArray **a, GPtrArray **b ) {
  GPtrArray *tmp;
  tmp = *a;
  *a = *b;
  *b = tmp;

}

void mutateIndV1(GRand *rnd, double mutateProb, double wcardProb, individual *myInd,
		   GArray *garraysL[NUM_HTABLES], 
		   GArray *garraysC[NUM_HTABLES][SUBH] ) {
  // TODO: stub for code
  gint i; // loop counter
  guint tmpChrome[NUM_GENE]; // tempChrome created
  time_stamp myT; // temporary time stamp data
  IPAddr myIP; // temporary IP address
  guint mySlot; // random slot number picked

  gdouble myRandD;
  


  
  // Copy the chromosome into the tmpChrome
  g_memmove( tmpChrome, myInd->chrome, 4 * NUM_GENE);

  myT.tot = tmpChrome[G_DURATION];
  //  time_stamp myT; // temporary time stamp data
  // IPAddr myIP; // temporary IP address  
  myT.byte[0] = 0xff; // not used, all should be -1

  // Hours, Minutes, Seconds
  for (i=1; i<4; i++) {
    myRandD = g_rand_double(rnd);
    if ( myRandD < mutateProb ) {
      mySlot = randslot(rnd, garraysC[G_DURATION][i]->len, wcardProb);
      myT.byte[i] = g_array_index (garraysC[G_DURATION][i], guchar, mySlot);
    }
  }

  tmpChrome[G_DURATION] = myT.tot; // duration

  // Service
  myRandD = g_rand_double(rnd);
  if ( myRandD < mutateProb ) {
    mySlot = randslot(rnd, garraysL[G_SERVICE]->len, wcardProb);
    tmpChrome[G_SERVICE] = g_array_index (garraysL[G_SERVICE], guint, mySlot);
  }

  // Source Port
  myRandD = g_rand_double(rnd);
  if ( myRandD < mutateProb ) {
    mySlot = randslot(rnd, garraysL[G_SOURCE_PORT]->len, wcardProb);
    tmpChrome[G_SOURCE_PORT] = g_array_index (garraysL[G_SOURCE_PORT], guint, mySlot);
  }

  // Dest Port
  myRandD = g_rand_double(rnd);
  if ( myRandD < mutateProb ) {
    mySlot = randslot(rnd, garraysL[G_DEST_PORT]->len, wcardProb);
    tmpChrome[G_DEST_PORT] = g_array_index (garraysL[G_DEST_PORT], guint, mySlot);
  }

  // Source IP xxx.xxx.xxx.xxx
  //            0   1   2   3
  myIP.full = tmpChrome[G_SRC_IP];
  for (i=0; i<4; i++) {
    myRandD = g_rand_double(rnd);
    if ( myRandD < mutateProb ) {
      mySlot = randslot(rnd, garraysC[G_SRC_IP][i]->len, wcardProb);
      myIP.octet[i] = g_array_index (garraysC[G_SRC_IP][i], guchar, mySlot);
    }
  }
  tmpChrome[G_SRC_IP] = myIP.full; // source IP

  // Dest IP xxx.xxx.xxx.xxx
  //          0   1   2   3
  myIP.full = tmpChrome[G_DEST_IP];
  for (i=0; i<4; i++) {
    myRandD = g_rand_double(rnd);
    if ( myRandD < mutateProb ) {
      mySlot = randslot(rnd, garraysC[G_DEST_IP][i]->len, wcardProb);
      myIP.octet[i] = g_array_index (garraysC[G_DEST_IP][i], guchar, mySlot);
    }
  }
  tmpChrome[G_DEST_IP] = myIP.full; // dest IP

  // Attack. Do not select wildcard, so put its probability at 0.
  myRandD = g_rand_double(rnd);
  if ( myRandD < mutateProb ) {
    mySlot = randslot(rnd, garraysL[G_ATTACK]->len, 0.0 );
    tmpChrome[G_ATTACK] = g_array_index (garraysL[G_ATTACK], guint, mySlot);
  }

  // Copy the temp chromosome into the individual
  g_memmove( myInd->chrome, tmpChrome, 4 * NUM_GENE);

}
